import { Request, Response } from 'express';
import { remindersService } from './service';
import fetch from 'node-fetch';

function hasFeature(vendor: any, featureKey: string): boolean {
  return vendor?.features?.includes(featureKey);
}

export const remindersController = {
  async create(req: any, res: Response) {
    const vendor = req.vendor;
    if (!hasFeature(vendor, 'reminders')) {
      return res.status(403).json({ error: 'Reminders are not available on your plan.' });
    }
    const { customer_id, type, use_ai, ai_input, manual_message, scheduled_at } = req.body;
    let message = manual_message;
    let generated_by_ai = false;
    if (use_ai) {
      // Call local AI endpoint
      const aiRes = await fetch(`${process.env.API_BASE_URL || 'http://localhost:3000'}/api/ai/generate-reminder`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': req.headers.authorization || ''
        },
        body: JSON.stringify(ai_input)
      });
      const aiData = await aiRes.json();
      message = aiData.message;
      generated_by_ai = true;
    }
    const { data, error } = await remindersService.createReminder({
      vendor_id: vendor.id,
      customer_id,
      message,
      type,
      generated_by_ai,
      scheduled_at: scheduled_at ? new Date(scheduled_at) : undefined
    });
    if (error) return res.status(400).json({ error: error.message });
    return res.status(201).json({ success: true, data });
  },

  async list(req: any, res: Response) {
    const vendor = req.vendor;
    if (!hasFeature(vendor, 'reminders')) {
      return res.status(403).json({ error: 'Reminders are not available on your plan.' });
    }
    const { data, error } = await remindersService.listReminders(vendor.id);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  async send(req: any, res: Response) {
    const vendor = req.vendor;
    if (!hasFeature(vendor, 'reminders')) {
      return res.status(403).json({ error: 'Reminders are not available on your plan.' });
    }
    const { id } = req.params;
    const { data: reminder, error } = await remindersService.getReminderById(id, vendor.id);
    if (error || !reminder) return res.status(404).json({ error: 'Reminder not found' });
    // Simulate send (log to console)
    console.log(`[REMINDER SENT] To customer ${reminder.customer_id}: ${reminder.message}`);
    const { data: updated, error: updateError } = await remindersService.markReminderSent(id, vendor.id);
    if (updateError) return res.status(400).json({ error: updateError.message });
    return res.json({ success: true, data: updated });
  }
}; 